<?php
include('includes/header.php'); 

$menu_id = isset($_GET['menu_id']) ? (int)$_GET['menu_id'] : 0;
$pdo = get_pdo();
$menu_details = null;
$menu_items = [];
$error = '';
$success = '';

// Fetch Menu Details
if ($menu_id && $pdo) {
    try {
        $stmt = $pdo->prepare("SELECT id, name FROM menus WHERE id = ?");
        $stmt->execute([$menu_id]);
        $menu_details = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$menu_details) {
            header('Location: menus.php?status=danger&msg=' . urlencode('Menu not found.'));
            exit;
        }

        // Fetch Menu Items (ordered by order_num)
        $stmt_items = $pdo->prepare("SELECT id, title, url, order_num, parent_id FROM menu_items WHERE menu_id = ? ORDER BY parent_id ASC, order_num ASC");
        $stmt_items->execute([$menu_id]);
        $menu_items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

    } catch (PDOException $e) {
        $error = 'Database error: ' . $e->getMessage();
    }
} else {
    header('Location: menus.php');
    exit;
}

// --- Handle Item Deletion ---
if (isset($_GET['action']) && $_GET['action'] == 'delete_item' && isset($_GET['item_id'])) {
    $item_id_to_delete = (int)$_GET['item_id'];
    try {
        // Delete item and its children (if using parent_id)
        $pdo->prepare("DELETE FROM menu_items WHERE id = ? OR parent_id = ?")->execute([$item_id_to_delete, $item_id_to_delete]);

        header('Location: menu_items.php?menu_id=' . $menu_id . '&status=success&msg=' . urlencode('Menu item deleted.'));
        exit;
    } catch (PDOException $e) {
        $error = 'Database error deleting item.';
    }
}

// --- Handle Item Addition (Simplified, No separate file for now) ---
if (isset($_POST['add_item'])) {
    $title = trim($_POST['item_title']);
    $url = trim($_POST['item_url']);
    $parent_id = empty($_POST['parent_id']) ? NULL : (int)$_POST['parent_id'];

    if (empty($title) || empty($url)) {
        $error = 'Title and URL are required for menu item.';
    } else {
        try {
            // Determine next order number (simple sequential ordering)
            $next_order = $pdo->query("SELECT IFNULL(MAX(order_num), 0) + 1 FROM menu_items WHERE menu_id = {$menu_id}")->fetchColumn();

            $stmt = $pdo->prepare("INSERT INTO menu_items (menu_id, title, url, parent_id, order_num) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$menu_id, $title, $url, $parent_id, $next_order]);

            header('Location: menu_items.php?menu_id=' . $menu_id . '&status=success&msg=' . urlencode('Menu item added successfully.'));
            exit;

        } catch (PDOException $e) {
            $error = 'Error adding menu item.';
        }
    }
}

// Display status message if redirected after action
if (isset($_GET['status']) && isset($_GET['msg'])) {
    $success = htmlspecialchars($_GET['msg']); // Use success here
}
?>

<h1 class="mb-4"><i class="fas fa-cogs me-2"></i> Manage Items for: **<?= htmlspecialchars($menu_details['name']) ?>**</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-4">
        <div class="card shadow mb-4">
            <div class="card-header py-3 bg-primary text-white">
                <h6 class="m-0 font-weight-bold">Add New Menu Item</h6>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="add_item" value="1">
                    <div class="mb-3">
                        <label for="item_title" class="form-label">Link Text</label>
                        <input type="text" class="form-control" id="item_title" name="item_title" required>
                    </div>
                    <div class="mb-3">
                        <label for="item_url" class="form-label">URL / Link</label>
                        <input type="text" class="form-control" id="item_url" name="item_url" placeholder="/page-slug or http://external.com" required>
                    </div>
                    <div class="mb-3">
                        <label for="parent_id" class="form-label">Parent Item (For Sub-menu)</label>
                        <select class="form-select" id="parent_id" name="parent_id">
                            <option value="">-- None (Top Level) --</option>
                            <?php 
                            // Only allow items that are currently top-level to be parents
                            foreach ($menu_items as $item): 
                                if (empty($item['parent_id'])):
                            ?>
                                <option value="<?= $item['id'] ?>"><?= htmlspecialchars($item['title']) ?></option>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary w-100"><i class="fas fa-plus me-1"></i> Add Item</button>
                </form>
            </div>
        </div>
    </div>

    <div class="col-lg-8">
        <div class="card shadow mb-4">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">Current Items</h6>
                <small class="text-muted">Use Drag & Drop (requires JS) or update manually.</small>
            </div>
            <div class="card-body">
                <?php if ($menu_items): ?>
                    <ul class="list-group" id="menu-item-list">
                        <?php 
                        // Simple nested list structure (requires further JS/PHP logic for full nesting)
                        foreach ($menu_items as $item): 
                            $is_child = !empty($item['parent_id']);
                        ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center <?= $is_child ? 'ms-4 bg-light' : '' ?>">
                            <div>
                                <i class="fas fa-arrows-alt me-2 text-muted"></i>
                                <?= $is_child ? '— ' : '' ?>
                                **<?= htmlspecialchars($item['title']) ?>** (Order: <?= $item['order_num'] ?>)
                                <br>
                                <small class="text-muted"><?= htmlspecialchars($item['url']) ?></small>
                            </div>
                            <div>
                                <a href="menu_item_edit.php?item_id=<?= $item['id'] ?>" class="btn btn-sm btn-info me-2" title="Edit Item"><i class="fas fa-edit"></i></a>
                                
                                <a href="menu_items.php?menu_id=<?= $menu_id ?>&action=delete_item&item_id=<?= $item['id'] ?>" 
                                   class="btn btn-sm btn-danger" 
                                   title="Delete Item"
                                   onclick="return confirm('Delete this menu item?');">
                                    <i class="fas fa-trash"></i>
                                </a>
                            </div>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                    <div class="mt-3 alert alert-warning">
                        **Note:** Sorting (drag & drop) requires a dedicated JavaScript library (like SortableJS) and AJAX backend processing, which is omitted in this basic raw PHP example.
                    </div>
                <?php else: ?>
                    <div class="alert alert-info">No items found in this menu. Add one using the form on the left.</div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<div class="mt-4">
    <a href="menus.php" class="btn btn-secondary"><i class="fas fa-arrow-left me-1"></i> Back to Menu List</a>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const itemList = document.getElementById('menu-item-list');
    
    // Check if the item list exists
    if (itemList) {
        new Sortable(itemList, {
            animation: 150,
            ghostClass: 'bg-info', // Visual class for the dragged item
            handle: '.fa-arrows-alt', // Only allow dragging from the icon
            
            // This function runs when an item is dropped
            onEnd: function (evt) {
                const itemOrder = [];
                // Collect the new order of item IDs
                itemList.querySelectorAll('li').forEach((li, index) => {
                    // Assuming you set data-id in the <li> tags, but here we'll parse the link
                    // Since we didn't add data-id, let's use a simpler approach: finding the edit link ID
                    const editLink = li.querySelector('a[title="Edit Item"]');
                    if (editLink) {
                        const urlParams = new URLSearchParams(editLink.href);
                        const itemId = urlParams.get('item_id');
                        
                        if (itemId) {
                            itemOrder.push({
                                id: itemId,
                                order: index + 1 // New order number starting from 1
                            });
                        }
                    }
                });

                // Send the new order to the server using Fetch/AJAX
                fetch('menu_reorder.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        menu_id: <?= $menu_id ?>,
                        order: itemOrder
                    }),
                })
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        console.log('Order updated successfully!');
                        // Optional: Show a temporary success message to the user
                    } else {
                        alert('Error saving order: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('AJAX Error:', error);
                    alert('Communication error. Could not save order.');
                });
            }
        });
    }
});
</script>
<?php include('includes/footer.php');