<?php
include('includes/header.php'); 

$error = '';
$success = '';

// 1. Get Menu ID from URL
if (!isset($_GET['id']) || empty($_GET['id'])) {
    // If no ID is provided, redirect back to user list
    header('Location: menus.php');
    exit;
}

$menu_id = (int)$_GET['id'];
$pdo = get_pdo();
$menu = null;

if ($pdo) {
    try {
        // Fetch current menu data
        $stmt = $pdo->prepare("SELECT id, name, location FROM menus WHERE id = ?");
        $stmt->execute([$menu_id]);
        $menu = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$menu) {
            // Menu not found
            header('Location: menus.php?status=danger&msg=' . urlencode('Menu not found.'));
            exit;
        }

    } catch (PDOException $e) {
        $error = 'Database error fetching menu details.';
    }
} else {
    $error = 'Database connection failed.';
}

// 2. Handle Form Submission (Update Logic)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $menu) {
    // Get updated data from form
    $new_name = trim($_POST['name']);
    $new_location = trim($_POST['location']);

    // --- Validation ---
    if (empty($new_name) || empty($new_location)) {
        $error = 'Menu Name and Location Slug are required.';
    } else {
        // --- Database Update ---
        try {
            // Check for duplicate location (excluding current menu)
            $check_stmt = $pdo->prepare("SELECT COUNT(*) FROM menus WHERE location = ? AND id != ?");
            $check_stmt->execute([$new_location, $menu_id]);

            if ($check_stmt->fetchColumn() > 0) {
                $error = 'The Location Slug already exists. Please choose a different one.';
            } else {
                // Update the menu
                $sql = "UPDATE menus SET name = ?, location = ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$new_name, $new_location, $menu_id]);

                $success = "Menu **" . htmlspecialchars($new_name) . "** updated successfully!";
                
                // Re-fetch updated data to display in the form
                $stmt = $pdo->prepare("SELECT id, name, location FROM menus WHERE id = ?");
                $stmt->execute([$menu_id]);
                $menu = $stmt->fetch(PDO::FETCH_ASSOC);
            }

        } catch (PDOException $e) {
            $error = 'Database error updating menu: ' . $e->getMessage();
        }
    }
}

// Fallback data for form (use POST data if available after a failed submission, else use fetched $menu)
$form_data = $_POST ? array_merge($menu, $_POST) : $menu;
?>

<h1 class="mb-4"><i class="fas fa-edit me-2"></i> Edit Menu: **<?= htmlspecialchars($menu['name'] ?? 'N/A') ?>**</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Edit Menu Details (ID: <?= $menu_id ?>)</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            <div class="mb-3">
                <label for="name" class="form-label">Menu Name <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="name" name="name" 
                       value="<?= htmlspecialchars($form_data['name'] ?? '') ?>" required>
            </div>
            
            <div class="mb-3">
                <label for="location" class="form-label">Location Slug <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="location" name="location" 
                       value="<?= htmlspecialchars($form_data['location'] ?? '') ?>" required>
                <small class="text-muted">Used by developers to call the menu (e.g., primary-nav, footer-links).</small>
            </div>

            <div class="d-flex justify-content-between">
                <a href="menus.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left me-1"></i> Back to Menu List
                </a>
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-sync-alt me-1"></i> Update Menu
                </button>
            </div>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>