<?php
include('includes/header.php'); 

$error = '';
$success = '';
$pdo = get_pdo();
$settings = []; // Array to hold key => value settings

// --- 1. Fetch Current Settings ---
if ($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM settings");
        $stmt->execute();
        
        // Convert the result into an associative array (key => value)
        $fetched_settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        $settings = $fetched_settings;

    } catch (PDOException $e) {
        $error = 'Database error fetching settings: ' . $e->getMessage();
    }
} else {
    $error = 'Database connection failed.';
}

// --- 2. Handle Form Submission (Update Logic) ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && $pdo) {
    
    // Define the list of fields we expect and process them
    $fields_to_update = [
        'site_name',
        'site_tagline',
        'admin_email'
    ];
    
    $update_count = 0;
    
    try {
        $pdo->beginTransaction();
        
        // Prepare statement for update/insert (UPSERT)
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) 
                               ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

        foreach ($fields_to_update as $key) {
            $value = trim($_POST[$key] ?? '');
            
            // Basic validation for email
            if ($key === 'admin_email' && !empty($value) && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                throw new Exception("Invalid email format for Admin Email.");
            }
            
            // Execute the UPSERT query
            $stmt->execute([$key, $value]);
            $update_count++;
        }
        
        $pdo->commit();
        $success = "General settings updated successfully ($update_count fields).";
        
        // Re-fetch updated settings
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM settings");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

    } catch (Exception $e) {
        $pdo->rollBack();
        $error = 'Error updating settings: ' . $e->getMessage();
    } catch (PDOException $e) {
        $pdo->rollBack();
        $error = 'Database error: ' . $e->getMessage();
    }
}

// Use $settings to pre-fill the form, falling back to POST data if submission failed
$form_data = $_POST ? array_merge($settings, $_POST) : $settings;
?>

<h1 class="mb-4"><i class="fas fa-cogs me-2"></i> General Settings</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3">
        <h6 class="m-0 font-weight-bold text-primary">Website Configuration</h6>
    </div>
    <div class="card-body">
        <form method="POST">
            
            <div class="mb-3">
                <label for="site_name" class="form-label">Site Name</label>
                <input type="text" class="form-control" id="site_name" name="site_name" 
                       value="<?= htmlspecialchars($form_data['site_name'] ?? '') ?>" required>
            </div>
            
            <div class="mb-3">
                <label for="site_tagline" class="form-label">Site Tagline/Description</label>
                <textarea class="form-control" id="site_tagline" name="site_tagline" rows="2"><?= htmlspecialchars($form_data['site_tagline'] ?? '') ?></textarea>
            </div>

            <div class="mb-3">
                <label for="admin_email" class="form-label">Admin Email Address</label>
                <input type="email" class="form-control" id="admin_email" name="admin_email" 
                       value="<?= htmlspecialchars($form_data['admin_email'] ?? '') ?>" required>
                <small class="text-muted">Used for official communications and site alerts.</small>
            </div>

            <button type="submit" class="btn btn-success mt-3">
                <i class="fas fa-save me-1"></i> Save Settings
            </button>
        </form>
    </div>
</div>

<?php include('includes/footer.php'); ?>