<?php
include('includes/header.php'); 

$error = '';
$success = '';
$pdo = get_pdo();

// --- 1. CSV Export Function ---
function export_to_csv($pdo, $table_name, $filename) {
    if (!$pdo) return false;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM $table_name");
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (empty($results)) return false;

        $file = fopen('php://temp', 'r+'); // Use temporary memory stream
        
        // Write header (column names)
        fputcsv($file, array_keys($results[0]));

        // Write data rows
        foreach ($results as $row) {
            fputcsv($file, $row);
        }

        rewind($file);
        $csv_data = stream_get_contents($file);
        fclose($file);

        // Send headers for download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        echo $csv_data;
        exit;

    } catch (PDOException $e) {
        return false;
    }
}

// --- Handle Export Action ---
if (isset($_GET['action']) && $_GET['action'] == 'export_csv' && isset($_GET['table'])) {
    $table = $_GET['table'];
    // Basic table name validation (add more robust validation in production)
    $allowed_tables = ['users', 'pages', 'menus', 'menu_items']; 
    if (in_array($table, $allowed_tables)) {
        export_to_csv($pdo, $table, $table . '_' . date('Ymd_His') . '.csv');
    } else {
        $error = 'Invalid table specified for export.';
    }
}

// --- 2. CSV Import Function (Basic Example for 'users' table) ---
if (isset($_POST['import_csv'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] == UPLOAD_ERR_OK) {
        
        $csv_file = $_FILES['csv_file']['tmp_name'];
        $handle = fopen($csv_file, 'r');
        $is_header = true;
        $imported_count = 0;

        try {
            $pdo->beginTransaction();
            // Assuming the CSV structure matches the 'users' table (username, email, role, password_plaintext)
            
            while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                if ($is_header) {
                    $is_header = false; // Skip header row
                    continue;
                }

                // Assuming columns: 0=username, 1=email, 2=role, 3=password (plaintext)
                if (count($data) >= 4) {
                    $username = trim($data[0]);
                    $email = trim($data[1]);
                    $role = trim($data[2] ?? 'user');
                    $password_hash = password_hash(trim($data[3]), PASSWORD_DEFAULT);

                    // Note: This simple example assumes INSERT. For existing users, use UPSERT or check first.
                    $stmt = $pdo->prepare("INSERT INTO users (username, email, role, password) VALUES (?, ?, ?, ?)");
                    $stmt->execute([$username, $email, $role, $password_hash]);
                    $imported_count++;
                }
            }
            
            $pdo->commit();
            $success = "Successfully imported $imported_count users from CSV.";
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            $error = 'CSV Import failed: ' . $e->getMessage();
        } finally {
            fclose($handle);
        }

    } else {
        $error = 'No CSV file uploaded or upload error occurred.';
    }
}


// --- 3. Database Backup (SQL Export - Requires custom function not fully shown here) ---
// Note: Generating a full SQL backup file in raw PHP is complex and often unreliable for large DBs. 
// A robust solution usually involves exec('mysqldump...') command or using a specialized library.
// For this example, we provide a basic template that you must implement securely.
if (isset($_GET['action']) && $_GET['action'] == 'backup_db') {
    $error = 'Full Database Backup functionality requires custom implementation using libraries or mysqldump command on the server, which is complex and risky in a raw PHP example without proper security. Please use phpMyAdmin for backups.';
}

?>

<h1 class="mb-4"><i class="fas fa-tools me-2"></i> Admin Tools & Utilities</h1>
<hr>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <?= $success ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <?= $error ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>

<div class="card shadow mb-4">
    <div class="card-header py-3 bg-warning text-white">
        <h6 class="m-0 font-weight-bold">CSV Import (Users)</h6>
    </div>
    <div class="card-body">
        <p>Upload a CSV file to bulk import data into the <code>users</code> table.</p>
        <form method="POST" enctype="multipart/form-data">
            <input type="hidden" name="import_csv" value="1">
            <div class="row">
                <div class="col-md-8">
                    <input type="file" class="form-control" name="csv_file" accept=".csv" required>
                    <small class="text-muted">Expected columns: Username, Email, Role, Password (plaintext).</small>
                </div>
                <div class="col-md-4">
                    <button type="submit" class="btn btn-warning w-100"><i class="fas fa-file-import me-1"></i> Import Users</button>
                </div>
            </div>
        </form>
    </div>
</div>

<div class="row">
    <div class="col-lg-6">
        <div class="card shadow mb-4">
            <div class="card-header py-3 bg-info text-white">
                <h6 class="m-0 font-weight-bold">Export Data to CSV</h6>
            </div>
            <div class="card-body">
                <p>Download a CSV backup of your table data.</p>
                <a href="tools.php?action=export_csv&table=users" class="btn btn-info btn-sm me-2"><i class="fas fa-download me-1"></i> Export Users</a>
                <a href="tools.php?action=export_csv&table=pages" class="btn btn-info btn-sm me-2"><i class="fas fa-download me-1"></i> Export Pages</a>
                <a href="tools.php?action=export_csv&table=menus" class="btn btn-info btn-sm"><i class="fas fa-download me-1"></i> Export Menus</a>
                </div>
        </div>
    </div>

    <div class="col-lg-6">
        <div class="card shadow mb-4">
            <div class="card-header py-3 bg-danger text-white">
                <h6 class="m-0 font-weight-bold">Database Backup (SQL)</h6>
            </div>
            <div class="card-body">
                <p>Create a full SQL backup of your entire database.</p>
                <a href="tools.php?action=backup_db" class="btn btn-danger w-100" onclick="return confirm('Full DB backup is complex. Are you sure you want to proceed?');">
                    <i class="fas fa-database me-1"></i> Backup Database (Advanced)
                </a>
                <small class="text-danger mt-2 d-block">Recommended: Use phpMyAdmin or host-provided tools for full database backups.</small>
            </div>
        </div>
    </div>
</div>

<?php include('includes/footer.php'); ?>